#!/bin/bash

# OffersExtractor Flask Deployment Script
# Run this script on your production server

set -e  # Exit on error

echo "🚀 Starting OffersExtractor deployment..."

# Configuration
APP_NAME="offers-extractor"
APP_DIR="/var/www/${APP_NAME}"
APP_USER="www-data"
PYTHON_VERSION="python3"
SERVICE_NAME="offers-extractor"

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}❌ Please run as root (use sudo)${NC}"
    exit 1
fi

echo -e "${YELLOW}📦 Step 1: Installing system dependencies...${NC}"
apt-get update
apt-get install -y python3 python3-pip python3-venv nginx supervisor certbot python3-certbot-nginx

echo -e "${YELLOW}📂 Step 2: Creating application directory...${NC}"
mkdir -p ${APP_DIR}
mkdir -p /var/log/${APP_NAME}

echo -e "${YELLOW}🔐 Step 3: Setting up permissions...${NC}"
chown -R ${APP_USER}:${APP_USER} ${APP_DIR}
chown -R ${APP_USER}:${APP_USER} /var/log/${APP_NAME}

echo -e "${YELLOW}🐍 Step 4: Creating virtual environment...${NC}"
cd ${APP_DIR}
sudo -u ${APP_USER} ${PYTHON_VERSION} -m venv venv

echo -e "${YELLOW}📥 Step 5: Installing Python dependencies...${NC}"
sudo -u ${APP_USER} ${APP_DIR}/venv/bin/pip install --upgrade pip
sudo -u ${APP_USER} ${APP_DIR}/venv/bin/pip install -r ${APP_DIR}/requirements.txt

echo -e "${YELLOW}⚙️ Step 6: Setting up environment file...${NC}"
if [ ! -f ${APP_DIR}/.env ]; then
    cp ${APP_DIR}/.env.example ${APP_DIR}/.env
    echo -e "${YELLOW}⚠️  Please edit ${APP_DIR}/.env and add your DEEPSEEK_API_KEY${NC}"
    read -p "Press Enter after editing .env file..."
fi

echo -e "${YELLOW}🔧 Step 7: Configuring Supervisor...${NC}"
cp ${APP_DIR}/supervisor.conf /etc/supervisor/conf.d/${SERVICE_NAME}.conf
supervisorctl reread
supervisorctl update

echo -e "${YELLOW}🌐 Step 8: Configuring Nginx...${NC}"
cp ${APP_DIR}/nginx.conf /etc/nginx/sites-available/extractor.stagi-edu.com

# Create symlink if it doesn't exist
if [ ! -L /etc/nginx/sites-enabled/extractor.stagi-edu.com ]; then
    ln -s /etc/nginx/sites-available/extractor.stagi-edu.com /etc/nginx/sites-enabled/
fi

# Test Nginx configuration
nginx -t

echo -e "${YELLOW}🔒 Step 9: Setting up SSL (Let's Encrypt)...${NC}"
read -p "Do you want to setup SSL now? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    certbot --nginx -d extractor.stagi-edu.com
fi

echo -e "${YELLOW}🚀 Step 10: Starting services...${NC}"
supervisorctl start ${SERVICE_NAME}
systemctl reload nginx

echo -e "${GREEN}✅ Deployment complete!${NC}"
echo ""
echo "📊 Service Status:"
supervisorctl status ${SERVICE_NAME}
echo ""
echo "🔍 Test the service:"
echo "  curl https://extractor.stagi-edu.com/health"
echo ""
echo "📝 View logs:"
echo "  tail -f /var/log/${APP_NAME}/error.log"
echo "  tail -f /var/log/${APP_NAME}/access.log"
echo ""
echo "🔄 Restart service:"
echo "  sudo supervisorctl restart ${SERVICE_NAME}"
